package com.base.browserwhite.ui.views

import android.animation.Animator
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import android.view.animation.LinearInterpolator
import com.base.browserwhite.R


class ColorProgress : View {

    private lateinit var ringPaint: Paint//圆环笔
    private var centerX = 0f
    private var centerY = 0f
    private var radius = 0f
    private var strokeWidth = 0f // 圆环的宽度
    private var margin = 10f
    private lateinit var ringRectF: RectF

    private lateinit var progressRectF: RectF
    private lateinit var progressPaint: Paint


    private var listProgressBean: List<ProgressBean> = listOf()
    private var currentAngle = 0f

    constructor(context: Context) : super(context) {
        initPaint(context)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        initPaint(context)
    }

    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        initPaint(context)
    }

    var finishAnimation: (() -> Unit)? = null

    private fun initPaint(context: Context) {
        strokeWidth = context.resources.getDimension(R.dimen.dp_9)

        ringPaint = Paint()
        ringPaint.isAntiAlias = true
        ringPaint.style = Paint.Style.STROKE
        ringPaint.setColor(Color.parseColor("#E6E7EC"))
        ringPaint.strokeWidth = strokeWidth

        progressPaint = Paint()
        progressPaint.isAntiAlias = true
        progressPaint.style = Paint.Style.STROKE
        progressPaint.strokeWidth = strokeWidth
        progressPaint.strokeCap = Paint.Cap.ROUND


        // 初始化圆的中心点和半径
        centerX = width / 2.0f
        centerY = height / 2.0f
        radius = (centerX.coerceAtMost(centerY) - strokeWidth / 2) - margin


    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        centerX = w / 2.0f
        centerY = h / 2.0f
        radius = (centerX.coerceAtMost(centerY) - strokeWidth / 2) - margin

        ringRectF = RectF(centerX - radius, centerY - radius, centerX + radius, centerY + radius)
        progressRectF = RectF(centerX - radius, centerY - radius, centerX + radius, centerY + radius)
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        // 绘制圆环
        canvas.drawOval(ringRectF, ringPaint)


        listProgressBean.forEachIndexed { index, progressBean ->
            progressPaint.setColor(Color.parseColor(progressBean.color))

            if (index != 0) {
                val lastProgressBean: ProgressBean = listProgressBean[index - 1]

                if (currentAngle <= progressBean.sweepAngle) {
                    canvas.drawArc(progressRectF, -90f, currentAngle, false, progressPaint)
                    progressBean.draw = true
                } else if (lastProgressBean.draw) {
                    canvas.drawArc(progressRectF, -90f, progressBean.sweepAngle, false, progressPaint)
                    progressBean.draw = true
                }
            } else {
                if (currentAngle >= progressBean.startAngle) {
                    canvas.drawArc(progressRectF, -90f, currentAngle, false, progressPaint)
                    progressBean.draw = true
                }
            }
        }


    }

    private fun setProgress(angle: Float) {
        currentAngle = angle
        invalidate()
    }

    fun animateProgress(list: List<ProgressBean>) {
        listProgressBean = list
        val totalAngle = list.maxOf { it.sweepAngle }
        val progressAnimator = ValueAnimator.ofFloat(0f, totalAngle)
        progressAnimator.interpolator = LinearInterpolator() // 平滑插值器
        progressAnimator.setDuration(1500) // 动画持续时间1000毫秒
        progressAnimator.addUpdateListener { animation ->
            val animatedValue = animation.getAnimatedValue() as Float
            setProgress(animatedValue)
        }
        progressAnimator.addListener(object : Animator.AnimatorListener {
            override fun onAnimationStart(animation: Animator) {
            }

            override fun onAnimationEnd(animation: Animator) {
                finishAnimation?.invoke()
            }

            override fun onAnimationCancel(animation: Animator) {
            }

            override fun onAnimationRepeat(animation: Animator) {
            }
        })
        progressAnimator.start()
    }

}

data class ProgressBean(
    val startAngle: Float = 0f,
    val sweepAngle: Float = 0f,
    var color: String = ""
) {
    var draw = false
}