package com.base.browserwhite.ui.activity.webbrowser

import android.annotation.SuppressLint
import android.net.Uri
import android.net.http.SslError
import android.view.View
import android.webkit.CookieManager
import android.webkit.JsResult
import android.webkit.SslErrorHandler
import android.webkit.ValueCallback
import android.webkit.WebChromeClient
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebSettings
import android.webkit.WebStorage
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.lifecycle.lifecycleScope
import com.base.browserwhite.bean.BookmarkBean
import com.base.browserwhite.bean.DownloadBean
import com.base.browserwhite.bean.HistoryBean
import com.base.browserwhite.databinding.FragmentWebViewBinding
import com.base.browserwhite.ui.activity.download.DownloadAdapter
import com.base.browserwhite.ui.fragment.BaseFragment
import com.base.browserwhite.ui.activity.download.DownloadDialog.showDownloadVideoDialog
import com.base.browserwhite.utils.ColorUtils
import com.base.browserwhite.utils.DownloadUtils.getDownloadJson
import com.base.browserwhite.utils.DownloadUtils.getDownloadJsonBean
import com.base.browserwhite.utils.DownloadUtils.saveDownloadRecordFile
import com.base.browserwhite.utils.LogEx
import com.base.browserwhite.utils.SpBeanUtils
import com.base.browserwhite.utils.SpBeanUtils.HISTORY_SP_KEY
import com.google.gson.Gson
import com.liulishuo.filedownloader.FileDownloader
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import okhttp3.Call
import okhttp3.Callback
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import org.json.JSONArray
import java.io.IOException
import kotlin.random.Random


class WebViewFragment : BaseFragment<FragmentWebViewBinding>() {

    private val TAG = "WebViewFragment"
    var url: String = ""
    private val downloadList = arrayListOf<DownloadBean>()
    var bookmarkBean: BookmarkBean = BookmarkBean()

    override val binding: FragmentWebViewBinding by lazy {
        FragmentWebViewBinding.inflate(layoutInflater)
    }

    var onPageFinished: ((uri: String?) -> Unit)? = null

    private var downloadAdapter: DownloadAdapter? = null

    override fun setView() {

        initWebSettings()
        loadWebView(url)
    }

    override fun setListener() {
        super.setListener()
        binding.tvReload.setOnClickListener {
            reloadWebView()
        }
        binding.webView.setOnScrollChangeListener { v, scrollX, scrollY, oldScrollX, oldScrollY ->
            if (scrollY == 0) {
                binding.refreshLayout.setEnabled(true)
            } else {
                binding.refreshLayout.setEnabled(false)
            }
        }
        binding.refreshLayout.setOnRefreshListener {
            lifecycleScope.launch {
                binding.refreshLayout.isRefreshing = true
                delay(Random.nextLong(1500, 2500))
                binding.refreshLayout.isRefreshing = false
            }
            reloadWebView()
        }

        binding.flDownload.setOnClickListener {
            if (downloadAdapter == null) {
                downloadAdapter = DownloadAdapter()
            }
            downloadAdapter?.let { adapter ->
                requireContext().showDownloadVideoDialog(
                    adapter, downloadList,
                    dismissAction = {
                        requireContext().saveDownloadRecordFile(downloadList)
                    },
                )
            }

        }
    }


    private fun reloadWebView() {
        binding.llError.visibility = View.GONE
        binding.webView.visibility = View.VISIBLE
        binding.progressBar.visibility = View.VISIBLE
        binding.progressBar.progress = 0
        binding.webView.reload()
    }

    fun loadWebView(loadUrl: String) {
        binding.llError.visibility = View.GONE
        binding.webView.visibility = View.VISIBLE
        LogEx.logDebug(TAG, "loadUrl=$loadUrl")
        binding.webView.loadUrl(loadUrl)
    }

    @SuppressLint("SetJavaScriptEnabled")
    fun initWebSettings() {
//        binding.webView.isFocusableInTouchMode = true
//        binding.webView.isFocusable = true

        val webSettings = binding.webView.settings
        webSettings.allowFileAccess = true// 设置允许访问文件数据
        webSettings.setSupportZoom(true)
        webSettings.builtInZoomControls = true
        webSettings.cacheMode = WebSettings.LOAD_NO_CACHE
        webSettings.domStorageEnabled = true
        webSettings.databaseEnabled = true
        @Suppress("DEPRECATION")
        webSettings.allowFileAccessFromFileURLs = true
        WebStorage.getInstance().deleteAllData()

        // 关键性代码，这里要给webView添加这行代码，才可以点击之后正常播放音频。记录一下。
        webSettings.mediaPlaybackRequiresUserGesture = false

        //设置WebView属性，能够执行Javascript脚本
        webSettings.javaScriptEnabled = true

        //设置混合内容模式：对于HTTPS和HTTP混合内容的加载，需要设置WebView以允许混合内容：
        webSettings.mixedContentMode = WebSettings.MIXED_CONTENT_ALWAYS_ALLOW

        //使用CookieManager设置允许跨域Cookie
        CookieManager.getInstance().setAcceptThirdPartyCookies(binding.webView, true)

        //适应屏幕设置
        webSettings.useWideViewPort = true
        webSettings.loadWithOverviewMode = true

        //网页使用localStorage等Web存储API，需要启用DOM存储
        webSettings.domStorageEnabled = true


        //设置WebChromeClient
        binding.webView.webChromeClient = object : WebChromeClient() {
            override fun onProgressChanged(view: WebView, newProgress: Int) {
                LogEx.logDebug(TAG, "onProgressChanged newProgress=$newProgress")

                binding.progressBar.progress = newProgress + 15
                if (binding.progressBar.progress >= 100) {
                    binding.progressBar.visibility = View.GONE
                }
                if (newProgress == 100) {
                    parseVideoLink(view)
                }
            }

            override fun onReceivedTitle(view: WebView?, title: String?) {
                super.onReceivedTitle(view, title)
            }

            override fun onShowFileChooser(
                webView: WebView?,
                filePathCallback: ValueCallback<Array<Uri?>?>,
                fileChooserParams: FileChooserParams?
            ): Boolean {
                return true
            }

            override fun onJsAlert(
                view: WebView?,
                url: String?,
                message: String?,
                result: JsResult?
            ): Boolean = true
        }
        binding.webView.webViewClient = object : WebViewClient() {
            override fun shouldOverrideUrlLoading(view: WebView?, request: WebResourceRequest?): Boolean {
                LogEx.logDebug(TAG, "shouldOverrideUrlLoading")
                val url = request?.url.toString()
                binding.llError.visibility = View.GONE
                bookmarkBean.url = url
                view?.loadUrl(url)
                return true
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                LogEx.logDebug(TAG, "onPageFinished")
                super.onPageFinished(view, url)
                val tittle = view?.title.toString()
                bookmarkBean.name = tittle
                onPageFinished?.invoke(url)

                //添加历史记录
                SpBeanUtils.addSpBean(
                    HISTORY_SP_KEY,
                    HistoryBean(
                        name = tittle, url = url ?: "",
                        time = System.currentTimeMillis(),
                        color = ColorUtils.getRandomColor()
                    )
                )
            }

            override fun onReceivedSslError(view: WebView?, handler: SslErrorHandler?, error: SslError?) {
                LogEx.logDebug(TAG, "onReceivedSslError")
                super.onReceivedSslError(view, handler, error)
                binding.webView.visibility = View.GONE
                binding.llError.visibility = View.GONE
                if (error != null) {
                    val domain = error.url
                    // 获取错误码
                    val primaryError = error.primaryError
                    // 根据错误码获取错误描述的中英文对照文本
                    val errorString = getSslErrorString(primaryError)
                    binding.tvErrorReason.text = errorString
                }
            }

            override fun onReceivedError(view: WebView?, request: WebResourceRequest?, error: WebResourceError?) {
                LogEx.logDebug(TAG, "onReceivedError ${error?.errorCode} ${error?.description}")
                super.onReceivedError(view, request, error)
                return
//                if (error?.errorCode == -1) {
//                    //-1 net::ERR_BLOCKED_BY_ORB 跨域读取不影响显示
//                    return
//                }
//                if (error?.errorCode == -6) {
//                    //-6 net::ERR_BLOCKED_BY_ORB 服务器拒绝连接，是服务的锅
//                    return
//                }
//                job?.cancel()
//                binding.webView.visibility = View.GONE
//                binding.llError.visibility = View.VISIBLE
//                if (error != null) {
//                    // 获取错误码
//                    val errorCode = error.errorCode
//                    // 获取错误描述
//                    val description = error.description
//
//                    // 创建错误信息文本
//                    val errorText = "Error ${errorCode}: $description"
//
//                    // 打印错误信息，或者根据需要进行其他处理
//                    binding.tvErrorReason.text = errorText
//                }
            }
        }

        //设置Cookie
        val instance = CookieManager.getInstance()
        instance.setAcceptThirdPartyCookies(binding.webView, true)

    }

    private var isParsing: Boolean = false

    @SuppressLint("NotifyDataSetChanged")
    fun parseVideoLink(view: WebView) {
        if (isParsing) return
        isParsing = true
        downloadList.clear()
        view.evaluateJavascript("(function() {" +
                "var videos = document.querySelectorAll('video');" +
                "var videoSources = [];" +
                "for (var i = 0; i < videos.length; i++) {" +
                "var video = videos[i];" +
                "var src = video.src;" +
                "videoSources.push(src);" +
                "}" +
                "return videoSources;" +
                "}).call(this);", ValueCallback<String?> { value -> // 这里处理获取到的视频链接数组
            LogEx.logDebug(TAG, "value=$value")
            // 将字符串转换为JSON数组，然后处理
            val jsonArray = JSONArray(value)

            val recordFile = requireContext().getDownloadJson()
            val olderList = getDownloadJsonBean(recordFile, Gson())
            olderList.forEach {
                it.status = FileDownloader.getImpl().getStatus(it.downloadId, it.path)
            }

            val set = hashSetOf<String>()
            for (i in 0 until jsonArray.length()) {
                val videoUrl = jsonArray.optString(i)
                if (videoUrl.startsWith("http") || videoUrl.startsWith("https")) {
                    set.add(videoUrl)
                }
            }
            if (set.size != 0) {
                binding.flDownload.visibility = View.VISIBLE
                binding.tvDownloadNumber.text = set.size.toString()
            }
            set.forEach { url ->
                LogEx.logDebug(TAG, url)
                val olderBean = olderList.findLast { it.url == url }
                if (olderBean != null) {
                    LogEx.logDebug(TAG, "old path=${olderBean.path} ${olderBean.downloadId}")
                    downloadList.add(olderBean.apply { uiType = 2 })
                } else {
                    downloadList.add(DownloadBean(url = url).apply { uiType = 2 })
                }
            }
            lifecycleScope.launch(Dispatchers.IO) {
                downloadList.forEach {
                    fastGetSize(it)
                }
                launch(Dispatchers.Main) {
                    downloadAdapter?.submitList(downloadList)
                    downloadAdapter?.notifyDataSetChanged()
                }
            }
            isParsing = false
        })

    }


    fun canGoBack(): Boolean {
        return binding.webView.canGoBack()
    }

    fun canGoForward(): Boolean {
        return binding.webView.canGoForward()
    }

    fun goBack() {
        return binding.webView.goBack()
    }

    fun goForward() {
        return binding.webView.goForward()
    }


    fun javascriptBack() {
        binding.webView.evaluateJavascript("window.history.back();", null)
    }


    private fun getSslErrorString(error: Int): String {
        return when (error) {
            SslError.SSL_DATE_INVALID -> "Date Invalid: The certificate's date is incorrect or the certificate has expired."
            SslError.SSL_EXPIRED -> "Expired: The SSL certificate has expired."
            SslError.SSL_IDMISMATCH -> "Hostname Mismatch: The hostname in the certificate does not match the requested hostname."
            SslError.SSL_INVALID -> "Invalid: The certificate is invalid."
            SslError.SSL_NOTYETVALID -> "Not Yet Valid: The certificate is not yet valid."
            SslError.SSL_UNTRUSTED -> "Untrusted: The certificate authority is not trusted."
            else -> "Unknown Error"
        }
    }

    private fun fastGetSize(bean: DownloadBean) = runCatching {
        if (bean.size == -1L) {
            val client = OkHttpClient()

            LogEx.logDebug(TAG, "fastGetSize ${bean.url}")
            val request = Request.Builder()
                .url(bean.url)
                .head() // 发送 HEAD 请求
                .build()

            try {
                client.newCall(request).enqueue(object : Callback {
                    override fun onFailure(call: Call, e: IOException) {
                        // 处理请求失败的情况
                        LogEx.logDebug(TAG, "onFailure")
                    }

                    override fun onResponse(call: Call, response: Response) {
                        if (response.isSuccessful) {
                            val contentLength = response.header("Content-Length") // 获取内容长度
                            contentLength?.let {
                                // 将内容长度转换为 Long 类型
                                val videoSize = it.toLong()
                                // 在这里使用视频大小，例如在主线程更新 UI 或存储到变量中
                                bean.size = videoSize
                                LogEx.logDebug(TAG, "videoSize=$videoSize")
                            }
                        }
                    }
                })
            } catch (e: Exception) {
                // 处理异常情况
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        requireContext().saveDownloadRecordFile(downloadList)
    }

}